#
#   Copyright (C) Stonesoft Corporation 2010 - 2012.
#   All rights reserved.
#
#   The StoneGate software, manuals, and technical
#   literature may not be reproduced in any form or
#   by any means except by permission in writing from
#   Stonesoft Corporation.
#
# maps evasions from predator4

module Predator4Module
  S_OPTION = "Option"
  S_INTEGER = "integer"
  # external codes (negative to avoid overlapping with predator4 codes)
  EVASION_TIMED_OUT = -1
  EVASION_SEGMENTATION_FAULT = -2
  EVASION_COULD_NOT_INTERPRET = -3

  S_0_SUCCESS = "0: Exploit succeeded."
  R_LF = /\n/
  R_SEGMENTATION_FAULT = /Segmentation fault/m

  class Attack
    attr_accessor :short_name, :name, :desc, :cve, :bid, :ms, :evasions, :evasions_by_name, :extra_options, :extra_options_by_name, :stages, :payloads
    def initialize( short_name )
      @short_name = short_name
      @name = name
      @desc = nil
      @cve = nil
      @bid = nil
      @ms = nil
      @evasions = []
      @evasions_by_name = {}

      @extra_options = []
      @extra_options_by_name = {}
      @stages = []
      @payloads = [] # attack payloads like 'clean', 'shell', 'fireworks'
    end
    def to_s
      "#{@name}"
    end

    def add_evasion( ev )
      @evasions.push ev
      @evasions_by_name[ ev.name ] = ev
    end
  end

  class Evasion
    attr_reader :name, :comment
    attr_accessor :options, :options_by_name
    def initialize(name, comment)
      @name = name
      @comment = comment
      @options = []
      @options_by_name = {}
    end
    def base_name
      return @name
    end
    def to_s
      "#{@name}"
    end

    def option_index( option )
      @options.size.times do |i|
        if @options[ i ] == option
          return i
        end
      end

      return nil
    end
  end

  class StageEvasion < Evasion
    attr_accessor :parent, :begin, :end
    def base_name
      return @parent.name
    end
  end

  class OriginalEvasion < Evasion
    attr_reader :supported_stages, :supported_stages_by_name, :stage_index
    def initialize(name, comment)
      super(name, comment)
      @supported_stages = []
      @supported_stages_by_name = {}
    end
    def overlap?(first, second)
      # overlaps if first evasions end is inside second or
      # second evasions end is inside first
      return ((@stage_index[first.begin] <= @stage_index[second.begin] and @stage_index[second.begin] < @stage_index[first.end]) or
              (@stage_index[second.begin] <= @stage_index[first.begin] and @stage_index[first.begin] < @stage_index[second.end]))
    end
    def generate_all_evasions
      evasions = []
      @stage_index = {}
      index = 0
      if @supported_stages.size > 0
        # check that the end stage is present
        last_stage = @supported_stages[-1]
        if last_stage.name != "end"
          @supported_stages.push SupportedStage.new("end", "autogenerated end")
        end
      end
      @supported_stages.each do |begin_stage|
        zone = false
        @supported_stages.each do |end_stage|
          if begin_stage == end_stage
            zone = true
            next
          end
          next if not zone
          name = "[" + begin_stage.name + "," + end_stage.name + "]" + @name
          evasion = StageEvasion.new(name, @comment)
          evasion.options = @options
          evasion.options_by_name = @options_by_name
          evasion.parent = self
          evasion.begin = begin_stage
          evasion.end = end_stage
          evasions.push evasion
        end
        @stage_index[begin_stage] = index
        index += 1
      end
      if @supported_stages.size == 0
        evasions.push self
      end
      return evasions
    end
  end

  class ExploitOption
    attr_accessor :name, :comment, :type

    def initialize( name, comment, type )
      @name = name
      @comment = comment
      @type = type
    end
  end

  class Option
    attr_reader :name, :comment
    def initialize(name, comment)
      @name = name
      @comment = comment
    end
    def to_s
      "#{@name}"
    end
    def get_value(index)
      return @enumerated_options[index]
    end
    def get_enum_size
      if @enumerated_options.size == 0
        raise "Enum size is nil for #{self.class}::#{self}"
      end
      return @enumerated_options.size
    end
  end

  class Preset
    attr_reader :name, :comment, :attacks
    def initialize(name, comment, attacks)
      @name = name
      @comment = comment
      @attacks = attacks
    end
    def to_s
      "#{@name}"
    end
  end

  class SupportedStage
    attr_reader :name, :comment
    def initialize(name, comment)
      @name = name
      @comment = comment
    end
    def to_s
      "#{name}"
    end
  end

  R_INTEGER_OPTION = /^\[ (\d+), (\d+) \] with step size (\d+)$/
  class IntegerOption < Option
    attr_reader :min, :max, :step
    # in a perfect world we'd like all of these
    # but there are situations where:
    # 10 from the beginning and 5 from the middle and 3 from the end will have to suffice
    MAX_ENTRIES = 18
    START_ENTRIES = 10
    END_ENTRIES = 3
    def initialize(name, comment, rest, all_options)
      super(name, comment)
      @all_options = all_options
      if rest.strip =~ R_INTEGER_OPTION
        @min = $1.to_i
        @max = $2.to_i
        @step = $3.to_i
      else
        raise "Unhandled IntegerOption rest #{rest}"
      end
      enum()
    end
    def to_s
      super + " [#{@min},#{@max}](#{@step})"
    end
    def get_value(index)
      if @all_options
        return @min + index*@step
      else
        value = @enumerated_options[index]
        if value.nil?
          value = rand(@area)*@step + @indent
        end
        return value
      end
    end

    def get_enum_size
      if @all_options
        return (@max-@min)/@step
      else
        super()
      end
    end

    def enum
      return nil if @all_options # do not attempt to map them
      if (@max-@min)/@step > MAX_ENTRIES
        result = []
        current = @min
        START_ENTRIES.times do |i|
          result.push current
          current += @step
        end
        current = @max
        END_ENTRIES.times do |i|
          result.push current
          current -= @step
        end
        (MAX_ENTRIES-result.size).times do
          result.push nil
        end
        @area = (@max/@step - END_ENTRIES) - (@min/@step + START_ENTRIES) + 1
        @indent = @min + START_ENTRIES*@step
      else
        result = []
        current = @min
        while(current < @max)
          result.push current
          current += @step
          break if result.size > MAX_ENTRIES
        end
      end
      @enumerated_options = result
      return nil
    end
  end
  R_CHOICE_SINGLE = /^\( single valid \):$/
  R_CHOICE_MULTIPLE = /^\( multiple valid \):$/
  class ChoiceOption < Option
    attr_reader :single_valid, :choices

    def initialize(name, comment, rest)
      super(name, comment)
      if rest.strip =~ R_CHOICE_SINGLE
        @single_valid = true
      elsif rest.strip =~ R_CHOICE_MULTIPLE
        @single_valid = false
      else
        raise "Unhandled ChoiceOption #{rest}"
      end
      @choices = {}
      enum()
    end

    def to_s
      super + " (#{@choices.keys.join("|")}) (single: #{@single_valid})"
    end

    def enum()
      if @single_valid
        @enumerated_options = @choices.keys
      else
        total = []
        ignore_first = true # it seems multiple valid does not mean empty
        (2**@choices.size).times do |i|
          if ignore_first
            ignore_first = false
            next
          end
          result = []
          @choices.size.times do |j|
            if(((i & (2**j))>>j) == 1)
              result.push @choices.keys[j]
            end
          end
          total.push result.join("|")
        end
        @enumerated_options = total
      end
      return nil
    end
  end

  A_PROPABILITY = ["25%", "50%", "75%", "1", "2", "3", "5", "8", "13", "21"]
  class PropabilityOption < Option
    def initialize(name, comment, all_options)
      super(name, comment)
      @all_options = all_options
      enum()
    end
    def enum()
      if @all_options
        @enumerated_options = ["1"]
      else
        @enumerated_options = A_PROPABILITY
      end
      return nil
    end
  end

  R_SPACE = /\s+/
  R_SPACESPACE = /\s\s+/
  class Parser
    def Parser.get_attacks(predator4 = "predator4")
      attack_names = []
      zone = false
      `#{predator4} --attacks 2>&1`.split(/\n/).each do |line|
        if line =~ /Available exploits:/
          zone = true
        elsif zone
          name, comment = line.strip.split(/-/, 2)
          attack_names.push name.strip
        end
      end
      return attack_names
    end
    def Parser.get_presets(predator4 = "predator4")
      zone = false
      presets = []
      `#{predator4} --presets 2>&1`.split(/\n/).each do |line|
        if line =~ /Available presets:/
          zone = true
        elsif zone
          name, comment = line.strip.split(R_SPACE, 2)
          comment =~ /(\d+)\sattacks$/
          presets.push Preset.new(name.strip, comment.strip, $1.to_i)
        end
      end
      return presets
    end
    def Parser.parse(selected_attack = nil, use_stages = true, predator4 = "predator4", all_options = false)
      attacks = []
      zone = false
      `#{predator4} --attacks 2>&1`.split(/\n/).each do |line|
        if line =~ /Available exploits:/
          zone = true
        elsif zone
          name, comment = line.strip.split(/-/, 2)
          if selected_attack.nil? or name.strip == selected_attack
            attacks.push Attack.new( name.strip )
          end
        end
      end

      # Extract extra info about attack
      zone = false
      zone2 = false
      zone3 = false
      attacks.each do |attack|
        zone_name = zone_id = zone_desc = zone_payload = zone_extraopt = zone_stages = false
        `#{predator4} --info=#{attack.short_name}`.split( /\n/ ).each do |line|

          if line =~ /Attack \"#{attack.short_name}\":/
            zone_name = true
          elsif zone_name
            attack.name = line
            zone_name = false
            zone_id = true
          elsif zone_id
            line.strip!
            if line.size() == 0
              next
            end

            if line =~ /^\w*CVE/
              attack.cve = line
            elsif line =~ /^\w*BID/
              attack.bid = line
            elsif line =~ /^\w*MS/
              attack.ms = line
            else
              attack.desc = line
              zone_id = false
              zone_desc = true
            end
          elsif zone_desc
            if line =~ /Payload support:/
              zone_desc = false
              zone_payload = true
            elsif line =~ /Exploit extra options:/
              zone_desc = false
              zone_extraopt = true
            elsif line =~ /Exploit stages:/
              zone_desc = false
              zone_stages = true
            else
              attack.desc << "\n#{line}"
            end
          elsif zone_payload
            if line =~ /Exploit extra options:/
              zone_payload = false
              zone_extraopt = true
            elsif line =~ /Exploit stages:/
              zone_payload = false
              zone_stages = true
            else
              if line =~ /^\s*(\S+)\s/
                payload = $1
                if not payload == "clean"
                  attack.payloads.push payload
                end
              end
            end
          elsif zone_extraopt
            if line =~ /Exploit stages:/
              zone_extraopt = false
              zone_stages = true
            else
              line.strip!
              if line.size() == 0
                next
              end

              if line =~ /([^\x20]*)\x20(.*)\x20-\x20\[([^\]]*)\]/
                exploit_opt = ExploitOption.new( $1, $2, $3 )
                attack.extra_options.push exploit_opt
                attack.extra_options_by_name[ $1 ] = exploit_opt
              else
                puts "Illegal extra option line: \"#{line}\""
              end
            end
          elsif zone_stages
            if line =~ /^\s*(\S+)\s/
              attack.stages.push $1
            end
          end
        end
      end

      # Extract evasions
      zone = false
      attacks.each do |attack|
        `#{predator4} --attack=#{attack.short_name} --evasions 2>&1`.split(/\n/).each do |line|
          line.strip!
          next if line.size == 0
          next if line == "100: Invalid parameters."
          if line =~ /Available evasions:/
            zone = true
          elsif zone
            name, comment = line.strip.split(/-/, 2)
            if comment.nil?
              puts "comment is nil for #{attack.short_name} from #{line}"
              comment=""
            end
            attack.add_evasion(OriginalEvasion.new(name.strip, comment.strip))
          end
        end
      end

      attacks.each do |attack|
        attack.evasions.each do |evasion|
          current_option = nil
          current_type = nil
          param_zone = false
          supported_stages_zone = false
          `#{predator4} --attack=#{attack.short_name} --evasion=#{evasion.name} 2>&1`.split(/\n/).each do |line|
            line.strip!
            next if line == "100: Invalid parameters."
            if line =~ /Parameters:/
              param_zone = true
            elsif line =~ /Supported stages:/
              param_zone = false
              supported_stages_zone = true
            elsif param_zone
              if line.strip.size == 0
                current_option = nil
                current_type = nil
              elsif current_option.nil? and current_type.nil?
                name, comment = line.strip.split(R_SPACESPACE, 2)
                if comment.nil?
                  comment = ""
                  #puts "#{attack.name} #{evasion.name} line #{line} does not have tabs"
                end
                current_option = [name.strip, comment.strip]
              elsif not current_option.nil? and current_type.nil?
                type, rest = line.strip.split(R_SPACE, 2)
                if type == "integer,"
                  option = IntegerOption.new(current_option[0], current_option[1], rest, all_options)
                elsif type == "Option"
                  option = ChoiceOption.new(current_option[0], current_option[1], rest)
                elsif current_option[0] == "Probability" and type == "Append"
                  option = PropabilityOption.new(current_option[0], current_option[1], all_options)
                elsif type == "Append"
                  puts "name has failed me >#{current_option[0]}<"
                else
                  raise "Unsupported option #{type}"
                end
                evasion.options.push option
                if evasion.options_by_name.has_key?(option.name)
                  raise "Evasion #{evasion.name} already has option #{option.name}"
                else
                  evasion.options_by_name[option.name] = option
                end
                current_type = true
              elsif evasion.options.last.class == IntegerOption
                raise "Unparsed line after integer option #{line}"
              elsif evasion.options.last.class == ChoiceOption
                name, comment = line.strip.split(/-/, 2 )
                name.strip!
                comment.strip!
                if evasion.options.last.choices.has_key?(name)
                  raise "Evasion #{evasion} option #{evasion.options.last} already has key #{name}"
                else
                  evasion.options.last.choices[name] = comment
                end
                evasion.options.last.enum
              elsif evasion.options.last.class == ProbabilityOption
                if line.strip == "Append '%' for probability (n%), otherwise run for every 'n'th invocation"
                else
                  raise "Unhandled probability option content #{line}"
                end
              else
                raise "Unsupported option type #{line}"
              end
            elsif use_stages and supported_stages_zone
              if not line.strip.size == 0
                name, comment = line.strip.split(/\s+/, 2)
                supported_stage = SupportedStage.new(name, comment)
                evasion.supported_stages.push supported_stage
                evasion.supported_stages_by_name[name] = supported_stage
              end
            end
          end
        end
      end
      attacks.each do |attack|
        original_evasions = attack.evasions
        attack.evasions = []
        original_evasions.each do |original_evasion|
          original_evasion.generate_all_evasions.each do |evasion|
            attack.evasions.push evasion
          end
        end
      end
      return attacks
    end
  end

  class NetworkConfig
    attr_accessor :iface, :src_ip, :dst_ip, :gw_ip, :mask
    def to_s
      str = "--if=#{@iface} --src_ip=#{@src_ip} --dst_ip=#{@dst_ip}"
      if not @gw_ip.nil?
        str += " --gw=#{@gw_ip}"
      end
      if not @mask.nil?
        str += " --src_prefix=#{@mask}"
      end
      return str
    end
  end

  # control interface
  class Interface
    def initialize(binary)
      @binary = binary
    end
    def get_options(attack, use_stages = true, all_options = false)
      return Parser.parse(attack, use_stages, @binary, all_options)[0]
    end
    def get_attacks()
      return Parser.get_attacks(@binary)
    end
    def clean(attack, network_config, src_port, port_usage)
      # test clean
      repeat = port_usage - 1
      result = ""
      cmd = ""
      code = 0
      explanation, result, cmd = nil
      repeat.times do |i|
        cmd = "#{@binary} #{network_config} --autoclose --attack=#{attack} --clean --src_port=#{src_port + i} 2>&1"
        code, explanation, result, cmd = command(cmd, 3, network_config.src_ip)
        log "#{i}/#{repeat}: cmd is #{cmd}", DEBUG
        log "#{i}/#{repeat}: result is #{result}", DEBUG
        if code == 0
          break
        end
        sleep 1 # this will throttle things
      end
    
      return code, explanation, result, cmd
    end
    R_DOT = /./
    R_SEGFAULT = /Segmentation fault/
    R_HTTP_SERVER_STOPPED = /HTTP server stopped/
    R_BODY_OPEN = /<body>/
    R_BODY_CLOSE = /<\/body>/
    R_SERVER_STARTED = /server started, visit (\S+) to get/
    R_SHASUM = /SHA-512 of content: (\S+)/
    def server_attack(attack, network_config, evasions, src_port, recdir, timeout, driver, randseed = nil, passthrough = [])
      cmd = "#{@binary} #{network_config} --autoclose --attack=#{attack} --src_port=#{src_port} --verifydelay=200 --obfuscate --extra=stay_open=true"
      if not randseed.nil?
        cmd += " --randseed=#{randseed}"
      end
      predator_output = ""
      recname = nil
      if not recdir.nil?
        recname = "#{recdir}/#{network_config.src_ip}:#{src_port}_#{time.hour}:#{time.min}:#{time.sec}:#{time.usec}.pcap"
        cmd += " --record=#{recname}"
      end
      
      evasions.each do |evasion, options|
        cmd += " --evasion=#{evasion.name}"
        evasion.options.each do |option|
          cmd += ",\"#{options[option.name]}\""
        end
      end
      if not passthrough.nil? and passthrough.size > 0
        passthrough.each do |pt|
          cmd += " #{pt}"
        end
      end
      p4serv = ATFExpect.new(cmd)
      str = p4serv.expect_line(R_SERVER_STARTED)
      predator_output += str
      if str =~ R_SERVER_STARTED
        url = $1
      end
      str = p4serv.expect_line(R_SHASUM)
      predator_output += str
      if str =~ R_SHASUM
        sha = $1
      end
      code = 299 # Timeout
      start_time = Time.new
      explanation = ""
      begin
        source = nil
        timeout(timeout) do
          if driver.nil?
            #source = `/root/wget-1.13.4/src/wget --tries=1 --quiet --timeout=#{(timeout/2).to_i} --output-document=- #{url} 2>&1`
            source = `wget --tries=1 --quiet --timeout=#{timeout - 2} --output-document=- #{url} 2>&1`
            source.force_encoding "ASCII-8BIT"
            if source =~ R_SEGFAULT
              code = -4
              explanation = "WGET SEGFAULT: #{cmd}"
            end
          else
            driver.navigate.to url
            source = driver.page_source
            source = source.split(R_BODY_CLOSE, 2)[0].to_s.split(R_BODY_OPEN, 2)[1].to_s
          end
        end
        if code > 0
          browser_sha = Digest::SHA512.hexdigest(source)
          if browser_sha == sha
            code = 0
            explanation = "Successful exploit"
          else
            code = 200
            explanation = "Different sha: #{sha} vs browser got #{browser_sha}"
          end
        end
      rescue Exception => e
        if e.class != Timeout::Error
          puts "#{e}\n#{e.backtrace.join("\n")}"
        end
      end
      predator_close(p4serv, cmd)
      return code, explanation, predator_output, cmd, recname
    end

    def predator_close(p4serv, cmd)
      begin
        timeout(2) do
          begin
            p4serv.send("Q\n")
            p4serv.flush
            p4serv.expect_line(R_HTTP_SERVER_STOPPED)
          rescue Errno::EIO => f
          rescue Timeout::Error => f
          end
        end
      rescue Exception => e
        if e.class != Timeout::Error
          puts "#{e.class}:#{e}\n#{e.backtrace.join("\n")}"
        end
      end
      p4serv.close
    end

    def attack(attack, network_config, evasions, src_port, attacker_shell_port, port_usage, recdir, timeout, randseed = nil, passthrough = [])
      # test clean
      code, explanation, result, cmd = clean(attack, network_config, src_port, port_usage)
      if code != 0
        return code + 1000, explanation, result, cmd
      end

      time = Time.new
      attack_src_port = src_port+port_usage-1
      
      if attacker_shell_port.nil?
        cmd = "#{@binary} #{network_config} --autoclose --attack=#{attack} --src_port=#{attack_src_port} --verifydelay=200 --obfuscate"
      else
        cmd = "#{@binary} #{network_config} --autoclose --attack=#{attack} --src_port=#{attack_src_port} --extra=bindport=#{attacker_shell_port} --verifydelay=200 --obfuscate"
      end
      if not randseed.nil?
        cmd += " --randseed=#{randseed}"
      end
      recname = nil
      if not recdir.nil?
        recname = "#{recdir}/#{network_config.src_ip}:#{attack_src_port}_#{time.hour}:#{time.min}:#{time.sec}:#{time.usec}.pcap"
        cmd += " --record=#{recname}"
      end
      
      evasions.each do |evasion, options|
        cmd += " --evasion=#{evasion.name}"
        evasion.options.each do |option|
          cmd += ",\"#{options[option.name]}\""
        end
      end
      if not passthrough.nil? and passthrough.size > 0
        passthrough.each do |pt|
          cmd += " #{pt}"
        end
      end
      
      code, explanation, result, cmd = command(cmd, timeout, network_config.src_ip)
      if not recname.nil?
        if File.exists?(recname)
          # add the code to the beginning of the filename
          dirname, basename = File.split(recname)
          new_recname = dirname + File::SEPARATOR + code.to_s + "_" + basename
          FileUtils.mv(recname, new_recname)
          recname = new_recname
        else
          recname = nil
        end
      end

      return code, explanation, result, cmd, recname
    end

    def command(cmd, timeout, attacker)
      log "cmd is #{cmd}", DEBUG
      result = ""
      begin
        timeout(timeout) do
          result = `#{cmd} 2>&1`
        end
      rescue Timeout::Error
        psout = `ps -afe | grep "predator4 " | grep #{attacker} | grep -v 'sh -c' | grep -v grep | grep -v ruby`.strip
        if psout.size > 0
          pid = psout.split(/\s+/)[1]
          #puts "killing #{pid} from #{psout}"
          result = "Pid #{pid} timed out - killed"
          `kill #{pid} 2>&1`
        else
          result = "Could not find pid for timed out.. this will cause problems for worker #{attacker}."
        end
        code = EVASION_TIMED_OUT
      end
      if code == EVASION_TIMED_OUT
        explanation = "Timed out"
      elsif result =~ R_SEGMENTATION_FAULT
        code = EVASION_SEGMENTATION_FAULT
        explanation = "Segmentation Fault"
      else
        if result.to_s.size > 0
          code, explanation = result.split(/\n/).last.split(/:/)
          if code =~ /^\d+$/ and not explanation.nil?
            code = code.to_i
            explanation.strip!
          else
            code = EVASION_COULD_NOT_INTERPRET
            explanation = "Could not interpret predator4 output"
          end
        else
          code = EVASION_COULD_NOT_INTERPRET
          explanation = "Empty predator4 output"
        end
      end
      log "result #{code}/#{explanation} from #{result}", DEBUG
      
      return code, explanation, result, cmd
    end
    def running?
      return (`ps -e | grep predator4 | grep -v grep`.strip.size > 0)
    end
  end
end

if __FILE__ == $0
  # 
  attacks = Predator4Module::Parser.parse("conficker")
  puts attacks.first.short_name
  presets = Predator4Module::Parser.get_presets()
  puts presets
  combinations = 1

  1.times do |i|
    attack_cmd = "predator4 --attack=conficker "
    attacks[0].evasions.each do |evasion|
      if true
        if evasion.name =~ /msrpc_ndrflag/
          puts evasion.name
          puts evasion.options.inspect
        end
        attack_cmd += "--evasion=#{evasion.name}"
        # evasion on
        evasion.options.each do |option|
          value = option.get_value(rand(option.get_enum_size()))
          attack_cmd += ",\"#{value}\""
        end
        attack_cmd += " "
      end
      if i == 0
        evasion.options.each do |option|
          #puts "#{option.name} #{option.enum}"
          combinations *= option.get_enum_size()
        end
      end
    end
    puts "#{i} #{attack_cmd}"
  end

  puts "total combo: #{combinations} " + sprintf("%0b", combinations).size.to_s
end
